//
//  ECGViewController.m
//  MicroLifeDeviceSDK-Demo
//
//  Created by willy.wu on 2021/8/25.
//  Copyright © 2021 WiFi@MBP. All rights reserved.
//

#import "ECGViewController.h"

@interface ECGViewController () <UIScrollViewDelegate>

@end

@implementation ECGViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    // Do any additional setup after loading the view.
    self.ECGView.showsVerticalScrollIndicator = NO;
    self.ECGView.showsHorizontalScrollIndicator = NO;
    [self.ECGView addSubview:self.ecg];
    
    self.showECGDataType = ECGDataTypeLPBPMovingAvgFilter;
    self.showECG = NO;
    
    kWeakSelf(self)
    [self connectElectrocardiography];
    [self.sdk cancelScan];
    [self.sdk device:@[self.electrocardiography]];
    [self.sdk getDidUpdateStateBlock:^(CBManagerState state) {
        NSString *log = [NSString stringWithFormat:@"DidUpdateState：%ld",(long)state];
        [weakself addLogWhitText:log];
    }];
    [self.sdk getScanDeviceBlock:^(NSInteger checkMacType, id  _Nonnull device) {
        NSString *log = [NSString stringWithFormat:@"ScanDevice Name：%@ UUID：%@ mac：%@ checkMacType：%ld",[device name],[device UUID],[device mac],checkMacType];
        [weakself addLogWhitText:log];
    } CancelScanBlock:^() {
        [weakself addLogWhitText:@"Cancel Scan"];
        [weakself.sdk startScan];
    }];
    [self.sdk getConnectDeviceStateBlock:^(id  _Nonnull device, CBCentralManager * _Nonnull central, CBPeripheral * _Nonnull peripheral, CBPeripheralState state, NSError * _Nonnull error) {
        [weakself connectDeviceInfo:device PeripheralState:state];
        switch (state) {
            case CBPeripheralStateConnected:
                [weakself connectedWork];
                break;
            case CBPeripheralStateDisconnected:
                [weakself.sdk startScan];
                break;
            default:
                break;
        }
    } CancelAllDevicesConnectionBlock:^{
        [weakself addLogWhitText:@"Cancel All Devices Connection"];
        weakself.settingValue.SW_continued_ECG = @(NO);
    }];
}

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
}

- (void)viewWillDisappear:(BOOL)animated {
    [super viewWillDisappear:animated];
}

- (void)viewDidLayoutSubviews {
    self.scrollView.contentSize = CGSizeMake(self.scrollView.frame.size.width, self.scrollView.frame.size.height*2.5);
}

- (void)connectedWork {
    [self.dataArray removeAllObjects];
    [self.beatArray removeAllObjects];
    [self.orderArray removeAllObjects];
    self.BPM = @(0);
    self.ECGDatas = 0;
    self.LDOFF = YES;
    [self drawRealTimeECGLine];
}

#pragma mark -
- (NSMutableArray *)dataArray {
    if (!_dataArray) {
        self.dataArray = [NSMutableArray array];
    }
    return _dataArray;
}

- (NSMutableArray *)beatArray {
    if (!_beatArray) {
        self.beatArray = [NSMutableArray array];
    }
    return _beatArray;
}

- (NSMutableArray *)orderArray {
    if (!_orderArray) {
        self.orderArray = [NSMutableArray array];
    }
    return _orderArray;
}

- (NSMutableArray *)lengthSource {
    if (!_lengthSource) {
        self.lengthSource = [NSMutableArray arrayWithObjects:@"N/A",@"30",@"60",@"540",@"1830", nil];
    }
    return _lengthSource;
}

- (CGRect)frameZero {
    CGFloat sEach = [UIScreen getPerMillimetreOfPT];
    NSInteger count = ceilf((SCREENWIDTH - 2*DDLeftPadding)/sEach);
    CGFloat width = count * sEach + 0.5;
    CGFloat height = 8 * sEach * 5 + 1;
    CGRect frame = CGRectMake(0, 0, width, height);
    return frame;
}

- (YOECGChartView *)ecg {
    if (!_ecg) {
        self.ecg = [[YOECGChartView alloc]initWithFrame:self.frameZero];
        self.ecg.standard.sampleFrequency = 256;
        [self.ecg reload];
    }
    return _ecg;
}

- (void)sampleFrequency:(int)sampleFrequency {
    if (self.dataArray.count == 0) {
        self.ecg.standard.sampleFrequency = sampleFrequency;
        [self.ecg reload];
    }
}

#pragma mark - device
- (void)connectElectrocardiography {
    kWeakSelf(self)
    self.electrocardiography = [MicroLifeECG shareWhithAuthorization:SDK_code Key:SDK_key Secret:SDK_secret];
    [self.electrocardiography addDeviceModel:@[@"EBPHKW2"]];
    [self.electrocardiography deviceConnectedBlock:^{
        [weakself.electrocardiography readSettingValues];
    }];
    [self.electrocardiography getValidationErrorBlock:^(NSString * _Nonnull item, NSString * _Nonnull info, id _Nonnull value) {
        [weakself addLogWhitText:[NSString stringWithFormat:@"item: %@\ninfo: %@\nvalue: %@",item,info,value]];
    }];
    [self.electrocardiography getReadDataModelBlock:^(NSInteger CMD, id  _Nonnull model, NSError * _Nonnull error) {
        switch (CMD) {
            case CMDECGReadECGRawData:
            case CMDECGReadECGHistoryRawData:
                weakself.showECG = YES;
                [weakself showECG:model];
                [weakself showBPM:model];
                break;
            case CMDECGReadDataAndList:
            case CMDECGReadLastData:
                [weakself readAllHistorys:model];
                break;
            case CMDECGReadDeviceUserSettingValues:
                [weakself readSettingValue:model];
                break;
            case CMDECGReadDeviceFunctionSettingValues:
                [weakself readDeviceFunctionSettingValues:model];
                break;
            case CMDECGReadDeviceCurrentStatus:
                [weakself logDeviceCurrentStatus:model];
                break;
            default:
                break;
        }
        if (weakself.showECG == NO) {
            [weakself log:weakself.electrocardiography CMD:CMD Model:model Error:error];
        }
    }];
}

- (void)logDeviceCurrentStatus:(MicroLifeECGDeviceCurrentStatus *)deviceCurrentStatus {
    NSString *log = @"MicroLife ECG Status: ";
    switch (deviceCurrentStatus.DeviceStatus) {
        case MicroLifeECGStatusManualEnterBluetooth:
            log = [log stringByAppendingFormat:@"Manual Enter Bluetooth"];
            break;
        case MicroLifeECGStatusDetectNewBloodPressureData:
            log = [log stringByAppendingFormat:@"Detect New Blood Pressure Data"];
            break;
        case MicroLifeECGStatusDetectNewBloodPressureDataWaitECGTrigger:
            log = [log stringByAppendingFormat:@"Detect New Blood Pressure Data Wait ECG Trigger"];
            break;
        case MicroLifeECGStatusOnlyECGModeWaitECGTrigger:
            log = [log stringByAppendingFormat:@"Only ECG Mode Wait ECG Trigger"];
            break;
        case MicroLifeECGStatusWaitLeadOnStable:
            log = [log stringByAppendingFormat:@"Wait Lead On Stable"];
            break;
        case MicroLifeECGStatusStartECGMeasurement:
            log = [log stringByAppendingFormat:@"Start ECG Measurement"];
            break;
        case MicroLifeECGStatusStopECGMeasurement:
            log = [log stringByAppendingFormat:@"Stop ECG Measurement"];
            break;
        case MicroLifeECGStatusSucceedAndStoreMemory:
            log = [log stringByAppendingFormat:@"Succeed And Store Memory"];
            break;
        case MicroLifeECGStatusFactoryMode:
            log = [log stringByAppendingFormat:@"Factory Mode"];
            break;
        default:
            break;
    }
    log = [log stringByAppendingFormat:@" ,ECGcode:%ld",deviceCurrentStatus.ECGcode];
    [self addLogWhitText:log];
}

- (void)readAllHistorys:(nonnull MicroLifeECGDRecord *)data {
    if ([data isKindOfClass:[MicroLifeECGDRecord class]]) {
        self.historyMeasuremeNumber.text = data.historyMeasurementNumber.stringValue;
    }
    [self addLogWhitText:[NSString stringWithFormat:@"MicroLifeECGDRecord:%@",data.parseDictionary]];
}

- (void)readSettingValue:(nonnull MicroLifeECGSettingValues *)settingValue {
    self.settingValue = settingValue;
    self.SWBT.on = self.settingValue.SW_BT.boolValue;
    self.SWBPplusECG.on = self.settingValue.SW_BPplusECG.boolValue;
    self.SWmax3003filter.on = self.settingValue.SW_max3003_filter.boolValue;
    self.SWTurnOffLeaddetect.on = self.settingValue.SW_turnOff_leaddetect.boolValue;
    self.SWcontinuedECG.on = self.settingValue.SW_continued_ECG.boolValue;
    self.SWBeepOn.on = self.settingValue.SW_BeepOn.boolValue;
    self.SWAfibOn.on = self.settingValue.SW_Afib.boolValue;
    self.MAMrest.text = [NSString stringWithFormat:@"%ld",
                                        self.settingValue.MAMRest];
    self.MAMInterval.text = [NSString stringWithFormat:@"%ld",self.settingValue.MAMInterval];
    NSString *length = @"N/A";
    switch (self.settingValue.length) {
        case MicroLifeSsampleTime30:
            length = @"30";
            break;
        case MicroLifeSsampleTime60:
            length = @"60";
            break;
        case MicroLifeSsampleTime540:
            length = @"540";
            break;
        case MicroLifeSsampleTime1830:
            length = @"1830";
            break;
        default:
            break;
    }
    self.length.text = length;
}

- (void)readDeviceFunctionSettingValues:(nonnull MicroLifeECGFunctionSettingValues *)functionSettingValues {
    self.functionSettingValues = functionSettingValues;
}

#pragma mark - show
- (void)showECG:(MicroLifeECGRawData *)model {
    NSMutableArray *arr = [NSMutableArray array];
    for (MicroLifeECGSampleData *data in model.ECGs) {
        NSNumber *rawdata = data.ECG;
        NSNumber *raw = [self filter:rawdata];
        [self.dataArray addObject:raw];
        NSInteger index = self.dataArray.count;
        BOOL checkAdd = (index%2 != 0);
        if (checkAdd) {
            [arr addObject:raw];
            [self.beatArray addObject:raw];
        }
    }
//    [self beat_classificationWithEcg:self.beatArray];
    [self.ecg drawRealTimeECGLine:arr];
}

- (void)showBPM:(MicroLifeECGRawData *)model {
    [self.orderArray addObject:model];
    switch (model.CMD) {
        case CMDECGReadECGRawData:
        case CMDECGReadECGHistoryRawData:
            if (self.BPM.integerValue != model.BPM.integerValue) {
                self.BPM = model.BPM;
                [self addLogWhitText:[NSString stringWithFormat:@"BPM：%@",self.BPM]];
            }
            if (self.LDOFF != model.LDOFF.boolValue) {
                self.LDOFF = model.LDOFF.boolValue;
                [self addLogWhitText:[NSString stringWithFormat:@"LDOFF：%@",self.LDOFF?@"EROOR":@"OK"]];
            }
            break;
//            if (self.LDOFF) {
//                [self addLogWhitText:[NSString stringWithFormat:@"Order >> %ld：%@",self.orderArray.count,model.Order]];
//                self.LDOFF = (self.orderArray.count == model.Order.integerValue+1);
//            }
            break;
        default:
            break;
    }
}

#pragma mark - AVG
- (NSNumber *)filter:(NSNumber *)rawdata {
    float raw = rawdata.floatValue;
    return @(raw);
}

- (NSArray *)loadFile:(NSString *)fileName {
    NSMutableArray *data = [NSMutableArray array];
    NSString *filepath = [[NSBundle mainBundle] pathForResource:fileName ofType:@"txt"];
    if (filepath) {
        NSError *error;
        NSString *contents = [NSString stringWithContentsOfFile:filepath encoding:NSUTF8StringEncoding error:&error];
        if (contents) {
            NSArray *array = [contents componentsSeparatedByString:@", "];
            for (NSString *item in array) {
                [data addObject:@([item doubleValue])];
            }
        } else {
            NSLog(@"loadFile Error");
        }
    } else {
        NSLog(@"No File");
    }
    return data;
}

#pragma mark - IBAction
- (IBAction)pdfAction:(id)sender {
    NSMutableArray *shares = [NSMutableArray array];
    if (self.ecg.pdfPath) {
        NSURL *pdfURL =[NSURL fileURLWithPath:self.ecg.pdfPath];
        [shares addObject:pdfURL];
    }
    for (UIImage *image in self.ecg.pdfImages) {
        [shares addObject:image];
    }
    if (shares.count > 0) {
        [self shareWithActivityItems:shares CallBlock:^(BOOL completed) {
            
        }];
    }
}

- (IBAction)comAction:(UIButton *)sender {
    self.showECG = NO;
    switch (sender.tag) {
        case 100:
            [self drawStaticECGLine];
            break;
        case 101:
            [self drawRealTimeECGLine];
            [self.ecg reload];
            [self file:self.electrocardiography.name];
            break;
        case 102:
            [self drawStaticECGPDF];
            break;
        case 200:
            [self.electrocardiography readDeviceStatus];
            break;
        case 201:
            self.showECG = YES;
            [self connectedWork];
            [self.electrocardiography startECGMeasurement];
            break;
        case 202:
            [self.electrocardiography stopECGMeasurements];
            break;
        case 300:
            [self.electrocardiography readAllHistorys];
            break;
        case 301:
            [self connectedWork];
            [self.electrocardiography readECGDataWithIndex:self.historyMeasuremeNumber.text.integerValue TransmitRate:TransmitRate512Hz];
            break;
        case 302:
            if (self.functionSettingValues) {
                NSInteger Func_Set1 = self.functionSettingValues.Fun_SEL_CUFFOK.integerValue | self.functionSettingValues.Fun_SEL_24hr.integerValue << 1 | self.functionSettingValues.Fun_SEL_Beep.integerValue << 2;
                NSInteger Func_Set2 = self.functionSettingValues.Fun_SEL_MAM.integerValue | self.functionSettingValues.Fun_SEL_SmartMAM.integerValue << 1 | self.functionSettingValues.Fun_SEL_MAMLIGHT.integerValue << 2 | self.functionSettingValues.Fun_SEL_IHB.integerValue << 3 | self.functionSettingValues.Fun_SEL_MAMAFib.integerValue << 4 | self.functionSettingValues.Fun_SEL_SingleAFib.integerValue << 5;
                NSInteger Func_Set3 = self.functionSettingValues.Fun_SEL_TrafficlightLCD.integerValue << 1 | self.functionSettingValues.Fun_SEL_TrafficlightLCD_TYPE.integerValue << 2 | self.functionSettingValues.Fun_SEL_MYBP.integerValue << 3 | self.functionSettingValues.Fun_SEL_MYCheck.integerValue << 4 | self.functionSettingValues.Fun_SEL_AllMEAN.integerValue << 5 | self.functionSettingValues.Fun_SEL_LCDShowAll.integerValue << 6 | self.functionSettingValues.Fun_SEL_MYBP_AVGday1.integerValue << 7;
                NSInteger Func_Set4 = self.functionSettingValues.Fun_SEL_RS232.integerValue | self.functionSettingValues.Fun_SEL_FacMeasure.integerValue << 1;
                [self.electrocardiography writeFunctionSettingValuesWithFunc_Set1:Func_Set1 Func_Set2:Func_Set2 Func_Set3:Func_Set3 TotalUserNumber:self.functionSettingValues.TotalUserNumber.integerValue MemorySet:self.functionSettingValues.MemorySet.integerValue OldPreDeflateValue:self.functionSettingValues.OldPreDeflateValue.integerValue YEARINIT:self.functionSettingValues.YEARINIT.integerValue Func_Set4:Func_Set4 Trafficlightlevel:self.functionSettingValues.TrafficLightLevel.integerValue BT_ModuleName:self.functionSettingValues.BTModuleName Fun_Sel_ValveType:self.functionSettingValues.FunSelValveType.integerValue ECG_Cal_Gain:100];
            }
            break;
        case 303:
            [self.electrocardiography clearAllHistorys];
            break;
        case 304:
            [self.electrocardiography disconnect];
            break;
        case 305:
            [self.electrocardiography readUserAndVersionData];
            break;
        case 306:
            [self.electrocardiography writeUserData:self.currUser.bpmUserID Age:self.currUser.age.integerValue];
            break;
        case 307:
            [self.electrocardiography readSettingValues];
            break;
        case 308:
            if (self.settingValue) {
                [self.electrocardiography writeSettingValuesWithSW_BT:self.settingValue.SW_BT.boolValue
                                                         SW_BPplusECG:self.settingValue.SW_BPplusECG.boolValue
                                                    SW_max3003_filter:self.settingValue.SW_max3003_filter.boolValue
                                                SW_turnOff_leaddetect:self.settingValue.SW_turnOff_leaddetect.boolValue // 新增參數
                                                     SW_continued_ECG:self.settingValue.SW_continued_ECG.boolValue
                                                            SW_BeepOn:self.settingValue.SW_BeepOn.boolValue
                                                              SW_Afib:self.settingValue.SW_Afib.boolValue // 新增參數
                                                              MAMRest:self.settingValue.MAMRest // 注意大小寫 (建議改為 MAMRest)
                                                               length:self.settingValue.length
                                                          MAMInterval:self.settingValue.MAMInterval];
            }  else {
                [self addLogWhitText:@"Please, readSettingValues ​​first"];
            }
            break;
        case 309:
            [self.electrocardiography readDeviceInfo];
            break;
        case 310:
            [self.electrocardiography readDeviceTime];
            break;
        case 311:
            [self.electrocardiography writeDeviceTime];
            break;
        case 312:
            [self.electrocardiography readSerialNumber];
            break;
        case 313:
            [self.electrocardiography readFunctionSettingValue];
            break;
        case 314:
            [self.electrocardiography readLastData];
            break;
        case 316:
            [self.electrocardiography askDeviceDisplayTransmitEndAnimation];
            break;
        default:
            break;
    }
}

- (IBAction)firfilterAction:(id)sender {
    [self.electrocardiography disconnect];
    [self.sdk startScan];
    NSArray *dataSource = @[@"RAW",@"LowPass_Filter",@"BandPass_Filter",@"Moving_Avg_Filter",@"LP_BP_Moving_Avg_Filter"];
    NSString *selectedItem = dataSource[self.showECGDataType];
    kWeakSelf(self)
    IdeabusPickView *ideabusPickView = [[IdeabusPickView alloc] initWithDataSource:dataSource selectedItem:selectedItem titleName:@"ShowECGDataType" confirmButtonName:@"Save" confirmBlock:^(id item) {
        [weakself firfilterTitle:item];
        weakself.showECGDataType = [dataSource indexOfObject:item];
    } cancelButtonName:NSLocalizedString(@"Cancel",nil) cancelBlock:^(id item) {
        
    } hideBlock:^{
        
    }];
    [ideabusPickView show];
}

- (void)firfilterTitle:(NSString *)item {
    kWeakSelf(self)
    dispatch_async(dispatch_get_main_queue(), ^{
        [weakself.Firfilter setTitle:item forState:UIControlStateNormal];
    });
}

- (void)drawStaticECGLine {
    float index = self.ecg.standard.oneGridSize*2;
    float width = (self.beatArray.count * self.ecg.standard.onePointWidth) + index;
    float height = self.ecg.frame.size.height + index;
    self.ECGView.contentSize = CGSizeMake(width, height);
    self.ECGView.scrollEnabled = YES;
    [self.ecg drawStaticECGLine:self.beatArray];
}

-(void)drawRealTimeECGLine {
    [self.dataArray removeAllObjects];
    [self.beatArray removeAllObjects];
    [self.ECGView setContentOffset:CGPointZero animated:NO];
    self.ECGView.scrollEnabled = NO;
    self.ecg.frame = self.frameZero;
    [self.ecg drawRealTimeECGLine:self.beatArray];
}

- (void)drawStaticECGPDF {
    float allWidth = self.beatArray.count * self.ecg.standard.onePointWidth;
    float ecgWidth = 5 * 10 * self.ecg.standard.oneGridSize;
    int indexH = ceil(allWidth/ecgWidth);
    float contentWidth = ecgWidth + self.ecg.standard.oneGridSize*2;
    float contentHeight = (self.ecg.frame.size.height * indexH) + (self.ecg.standard.oneGridSize * 3 * indexH);
    self.ECGView.contentSize = CGSizeMake(contentWidth, contentHeight);
    self.ECGView.scrollEnabled = YES;
//    self.ecg.statictiveMaxNum = 30;
    [self.ecg drawStaticECGLine:self.beatArray oneLine:NO];
}

- (IBAction)switchSettingValues:(UISwitch *)sender {
    NSString *log = [NSString stringWithFormat:@"Switch SettingValues:"];
    switch ([sender tag]) {
        case 100:
            log = [log stringByAppendingFormat:@"SW_BT"];
            self.settingValue.SW_BT = @(sender.on);
            break;
        case 101:
            log = [log stringByAppendingFormat:@"SW_BPplusECG"];
            self.settingValue.SW_BPplusECG = @(sender.on);
            break;
        case 102:
            log = [log stringByAppendingFormat:@"SW_max3003_filter"];
            self.settingValue.SW_max3003_filter = @(sender.on);
            break;
        case 103:
            log = [log stringByAppendingFormat:@"SW_turnOff_leaddetect"];
            self.settingValue.SW_turnOff_leaddetect = @(sender.on);
            break;
        case 104:
            log = [log stringByAppendingFormat:@"SW_continued_ECG"];
            self.settingValue.SW_continued_ECG = @(sender.on);
            break;
        case 107:
            log = [log stringByAppendingFormat:@"SW_BeepOn"];
            self.settingValue.SW_BeepOn = @(sender.on);
            break;
        case 108:
            log = [log stringByAppendingFormat:@"SW_Afib"];
            self.settingValue.SW_Afib = @(sender.on);
            break;
        default:
            break;
    }
    log = [log stringByAppendingFormat:@" %@",sender.on?@"YES":@"NO"];
    [self addLogWhitText:log];
}

- (void)editSettingValues:(UITextField *)sender {
    NSString *log = [NSString stringWithFormat:@"Edit SettingValues:"];
    switch ([sender tag]) {
        case 105:
            log = [log stringByAppendingFormat:@"MAMInterval: %@",sender.text];
            self.settingValue.MAMInterval = sender.text.integerValue;
            break;
        case 108:
            log = [log stringByAppendingFormat:@"MAMrest: %@",sender.text];
            self.settingValue.MAMRest = sender.text.integerValue;
            break;
        case 106:
        {
            log = [log stringByAppendingFormat:@"length %@",sender.text];
            SsampleTime length = MicroLifeSsampleTimeNA;
            switch ([self.lengthSource indexOfObject:sender.text]) {
                case 1:
                    length = MicroLifeSsampleTime30;
                    break;
                case 2:
                    length = MicroLifeSsampleTime60;
                    break;
                case 3:
                    length = MicroLifeSsampleTime540;
                    break;
                case 4:
                    length = MicroLifeSsampleTime1830;
                    break;
                default:
                    break;
            }
            self.settingValue.length = length;
        }
            break;
        default:
            break;
    }
    [self addLogWhitText:log];
}

- (IBAction)editMAMInterval:(UITextField *)sender {
    [self.view endEditing:YES];
    NSMutableArray *dataSource = [NSMutableArray arrayWithObjects:@"15", @"60", nil];
    IdeabusPickView *ideabusPickView = [[IdeabusPickView alloc] initWithDataSource:dataSource selectedItem:sender.text titleName:@"Interval" confirmButtonName:NSLocalizedString(@"Save",nil) confirmBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } cancelButtonName:NSLocalizedString(@"Cancel",nil) cancelBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } hideBlock:^{
        
    }];
    [ideabusPickView show];
}

- (IBAction)editMAMrest:(UITextField *)sender {
    [self.view endEditing:YES];
    NSMutableArray *dataSource = [NSMutableArray arrayWithObjects:@"0", @"60", @"180", @"300", nil];
    IdeabusPickView *ideabusPickView = [[IdeabusPickView alloc] initWithDataSource:dataSource selectedItem:sender.text titleName:@"Rest" confirmButtonName:NSLocalizedString(@"Save",nil) confirmBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } cancelButtonName:NSLocalizedString(@"Cancel",nil) cancelBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } hideBlock:^{
        
    }];
    [ideabusPickView show];
}

- (IBAction)editLength:(UITextField *)sender {
    [self.view endEditing:YES];
    IdeabusPickView *ideabusPickView = [[IdeabusPickView alloc] initWithDataSource:self.lengthSource selectedItem:sender.text titleName:@"Length" confirmButtonName:NSLocalizedString(@"Save",nil) confirmBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } cancelButtonName:NSLocalizedString(@"Cancel",nil) cancelBlock:^(id item) {
        sender.text = item;
        [self editSettingValues:sender];
    } hideBlock:^{
        
    }];
    [ideabusPickView show];
}
/*
#pragma mark - Navigation

// In a storyboard-based application, you will often want to do a little preparation before navigation
- (void)prepareForSegue:(UIStoryboardSegue *)segue sender:(id)sender {
    // Get the new view controller using [segue destinationViewController].
    // Pass the selected object to the new view controller.
}
*/

@end
