//
//  BLESDK.h
//  MicroLifeDeviceSDK
// 1. Scan and connect with Bluetooth device.
// 2. The command is sent.
// 3. Reply to receive.
//  Created by willy.wu on 2020/11/18.
//

#import <Foundation/Foundation.h>
#import <CoreBluetooth/CoreBluetooth.h>
#import <MicroLifeDeviceSDK/BluetoothManagerProtocol.h>

NS_ASSUME_NONNULL_BEGIN

@interface BLESDK : NSObject <BluetoothManagerProtocol>

+ (instancetype)shareOne;

+ (instancetype)share;

/// show Log
/// @param showLog showLog
- (void)showLog:(BOOL)showLog;

/// Set up Bluetooth
/// @param bluetooth MicroLifeDevice Class
- (void)device:(NSArray *)bluetooth;

/// Set up Model
/// @param model Model Name
- (void)model:(NSArray *)model;

/// Set Sacn stop Time
/// @param time time
- (void)stopTime:(NSInteger)time;

/// Set RSSI
/// @param rssi RSSI
- (void)rssi:(NSInteger)rssi;

/// When CBManagerStatePoweredOn automatically opens scanning [Default: Open]
/// @param autoScan autoSca
- (void)autoScan:(BOOL)autoScan;

/// Get the currently connected devices
- (NSArray *)findConnectedDevices;

/// Set AutoReconnect
/// @param device device
- (void)setAutoReconnect:(id)device;

/// Cancel AutoReconnect
/// @param device device
- (void)cancelAutoReconnect:(id)device;

#pragma mark - BluetoothManagerProtocol
/**
 * 開始掃描藍牙設備
 * Start scanning for bluetooth devices
 */
- (void)startScan;

/**
 * 停止掃描藍牙設備
 * Stop scanning for bluetooth devices
 */
- (void)cancelScan;

/**
 * 連接指定的藍牙設備
 * Connect to specified bluetooth device
 * @param device 要連接的設備 (device to connect)
 */
- (void)connectDevice:(id)device;

/**
 * 斷開指定設備的連接
 * Disconnect specified device
 * @param device 要斷開的設備 (device to disconnect)
 */
- (void)disconnectDevice:(id)device;

/**
 * 斷開所有已連接的設備
 * Disconnect all connected devices
 */
- (void)cancelAllConnect;

/**
 * 設置特徵值通知
 * Set characteristic notification
 * @param device 目標設備 (target device)
 * @param characteristic 目標特徵值 (target characteristic)
 */
- (void)setNotifiy:(id)device characteristic:(CBCharacteristic *)characteristic;

/**
 * 寫入數據到設備
 * Write data to device
 * @param value 要寫入的數據 (data to write)
 * @param deviceInfo 目標設備信息 (target device info)
 * @return 是否寫入成功 (whether write successful)
 */
- (BOOL)writeValue:(NSString *)value DeviceInfo:(id)deviceInfo;

/**
 * 心跳檢測
 * Heart beat check
 * @param value 心跳數據 (heart beat data)
 * @param deviceInfo 目標設備信息 (target device info)
 */
- (void)HEARTCHECK:(NSString *)value DeviceInfo:(id)deviceInfo;

/**
 * 設置藍牙狀態更新回調
 * Set bluetooth state update callback
 */
- (void)getDidUpdateStateBlock:(didUpdateStateBlock)block;

/**
 * 設置掃描相關回調
 * Set scanning related callbacks
 */
- (void)getScanDeviceBlock:(scanDeviceBlock)scanDeviceBlock
           CancelScanBlock:(cancelScanBlock)cancelScanBlock;

/**
 * 設置連接相關回調
 * Set connection related callbacks
 */
- (void)getConnectDeviceStateBlock:(connectDeviceStateBlock)connectDeviceStateBlock
   CancelAllDevicesConnectionBlock:(cancelAllDevicesConnectionBlock)cancelAllDevicesConnectionBlock;

/**
 * 設置服務發現回調
 * Set service discovery callback
 */
- (void)getDiscoverServicesBlock:(discoverServicesBlock)block;

/**
 * 設置特徵值發現回調
 * Set characteristic discovery callback
 */
- (void)getDiscoverCharacteristicsBlock:(discoverCharacteristicsBlock)block;

/**
 * 設置描述符發現回調
 * Set descriptor discovery callback
 */
- (void)getDiscoverDescriptorsForCharacteristicBlock:(discoverDescriptorsForCharacteristicBlock)block;

/**
 * 設置特徵值通知狀態更新回調
 * Set characteristic notification state update callback
 */
- (void)getUpdateNotificationStateForCharacteristicBlock:(updateNotificationStateForCharacteristicBlock)block;

/**
 * 設置讀取RSSI回調
 * Set read RSSI callback
 */
- (void)getReadRSSIBlock:(readRSSIBlock)block;

/**
 * 設置寫入特徵值回調
 * Set write characteristic callback
 */
- (void)getWriteValueForCharacteristicBlock:(writeValueBlock)block;

/**
 * 設置讀取特徵值回調
 * Set read characteristic callback
 */
- (void)getReadValueForCharacteristicBlock:(readValueBlock)block;

/**
 * 設置讀取描述符回調
 * Set read descriptor callback
 */
- (void)getReadValueForDescriptorsBlock:(readValueForDescriptorsBlock)block;

/**
 * 設置特徵值通知數據回調
 * Set characteristic notification data callback
 */
- (void)getReadNotifyValueForCharacteristicBlock:(readNotifyValueForCharacteristicBlock)block;

@end

NS_ASSUME_NONNULL_END
